/***************************************************************
 *                 Mathematical Object Library                 *
 *           Conversion tools Library - Include File           *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.6.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003,2004,2005,2006,2007,2008,2009 COLLARD Christophe
 * copyright © 2003,2004,2005,2006,2007,2008,2009 Centre National de la Recherche Scientifique
 * copyright © 2003,2004,2005,2006,2007,2008,2009 Arts et Métiers ParisTech
 * copyright © 2003,2004,2005,2006,2007 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::conversion_tools
    \brief conversion tools library for basic mathematical objects \n

    \htmlonly 
    <FONT color="#838383">

    conversion_tools belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006, 2007, 2008, 2009 Christophe COLLARD \n
            copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2007, 2008, 2009 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2007, 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	    copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.6.0
    \date 2003-2011
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type conversion tools
#endif

#ifndef _conversion_tools_h
#define _conversion_tools_h

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__PARAMETERS_H)
#include "parameters.h"
#endif

#if !defined(__MATHS_H)
#include "MOL++/maths.h"
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__SYMTENSORS2_H)
#include "MOL++/symtensors2.h"
#endif

#if !defined(__TENSORS4_H)
#include "MOL++/tensors4.h"
#endif

using namespace std;

namespace mol
{


//====================
class conversion_tools
//====================
{
  public :
    template <class T> friend matrix<T>     vector2matrix    (const vector<T>&, int, int);
    template <class T> friend symtensor2<T> vector2symtensor (const vector<T>&);
    template <class T> friend vector<T>     matrix2vector    (const matrix<T>&);
    template <class T> friend tensor4<T>    matrix2tensor    (const matrix<T>&, int, int, int, int);
    template <class T> friend tensor4<T>    matrix2symtensor (const matrix<T>&);
    template <class T> friend matrix<T>     tensor2matrix    (const tensor4<T>&);
    template <class T> friend vector<T>     symtensor2vector (const symtensor2<T>&);
    template <class T> friend matrix<T>     symtensor2matrix (const tensor4<T>&);
};


//=====Public methods for conversion tools===================================


/*!
  \brief Converts a vector into a matrix.

  We define the matrix \f$ M \in \mathbb{R}^p \times \mathbb{R}^q \f$ through the coordinates of the vector \f$ v \in \mathbb{R}^{pq} \f$ by \n
  \f$ M = \left( \begin{matrix}
  v_1 & v_2 & v_3 & \dots & v_q \\
  v_{q+1} & v_{q+2} & v_{q+3} & \dots & v_{2q} \\
  \vdots & \vdots & \vdots & \ddots & \vdots \\
  v_{(p-1)q+1} & v_{(p-1)q+2} & v_{(p-1)q+3} & \cdots & v_{pq}
  \end{matrix} \right) \f$.

  \param v vector
  \param rows number of matrix rows : p
  \param columns number of matrix columns : q
  \return M
*/

//------------------------------------------------------------------------------------
template <class T> matrix<T> vector2matrix (const vector<T>& v, int rows, int columns)
//------------------------------------------------------------------------------------
{
  assert (v.dim() == rows*columns);

  matrix<T> mat(rows,columns,false);

  for (int i=1,p=1; i<=rows; i++)
    for (int j=1; j<=columns; j++)
      mat(i,j) = v[p++];

  return mat;
}


/*!
  \brief Converts a vector into a symmetric \f$ 2^\text{nd} \f$ order tensor.

  Let \f$ v \f$ be a vector of \f$ \mathbb{R}^n \f$. We define \f$ T \in \mathbb{R}^m \times \mathbb{R}^m \f$, with \f$ m = \frac{1}{2} (\sqrt{1+8n} - 1) \f$, by
  \f$ T = \left( \begin{matrix}
      T_{11} = v_1 & T_{12} = \frac{1}{\sqrt{2}} v_{n+1} & T_{13} = \frac{1}{\sqrt{2}} v_{n+2} & \dots & T_{1n} = \frac{1}{\sqrt{2}} v_{2n-1} \\ \\
      T_{12} & T_{22} = v_2 & T_{23} = \frac{1}{\sqrt{2}} v_{2n} & \dots & T_{2n} = \frac{1}{\sqrt{2}} v_{3n-3} \\ \\
      T_{13} & T_{23} & T_{33} = v_3 & \dots & T_{3n} = \frac{1}{\sqrt{2}} v_{4n-6} \\ \\
      \vdots & \vdots & \vdots & \ddots & \vdots \\ \\
      T_{1n} & T_{2n} & T_{3n} & \dots & T_{nn} = v_n
      \end{matrix} \right) \f$.

  \param v vector
  \return \f$ T \f$
  \sa symtensor2vector
*/

//--------------------------------------------------------------------
template <class T> symtensor2<T> vector2symtensor (const vector<T>& v)
//--------------------------------------------------------------------
{
  assert (v.dim());

  long double factor = .5 * sqrt( (long double) 2);
  int n = .5 * (sqrt(1+8*v.dim()) - 1);
  symtensor2<T> tsr(n,n,false);

  for (int i=1,p=1; i<=n; i++)
    { tsr(i,i) = v[i];
      for (int j=i+1; j<=n; j++, p++)
	tsr(j,i) = factor * v[n+p];
    }

  return tsr;
}


/*!
  \brief Converts a matrix into a vector.

  We define the vector \f$ v \in \mathbb{R}^{pq} \f$ through the coordinates of the matrix \f$ M \in \mathbb{R}^p \times \mathbb{R}^q \f$ by \n
  \f$ M = \left( \begin{matrix}
  v_1 & v_2 & v_3 & \dots & v_q \\
  v_{q+1} & v_{q+2} & v_{q+3} & \dots & v_{2q} \\
  \vdots & \vdots & \vdots & \ddots & \vdots \\
  v_{(p-1)q+1} & v_{(p-1)q+2} & v_{(p-1)q+3} & \cdots & v_{pq}
  \end{matrix} \right) \f$.

  \param mat matrix M
  \return v
*/

//---------------------------------------------------------------
template <class T> vector<T> matrix2vector (const matrix<T>& mat)
//---------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  vector<T> v(mat.Rows()*mat.Columns(),false);

  for (int i=1,p=1; i<=mat.Rows(); i++)
    for (int j=1; j<=mat.Columns(); j++)
      v[p++] = mat(i,j);

  return v;
}


/*!
  \brief Converts a matrix into a \f$ 4^\text{th} \f$ order tensor.

  Let  \f$ M \f$ be a matrix in \f$ \mathbb{R}^{nm} \times \mathbb{R}^{pq} \f$. We define the \f$ 4^\text{th} \f$ order tensor \f$ C \f$ in \f$ \mathbb{R}^n \times \mathbb{R}^m \times \mathbb{R}^p \times \mathbb{R}^q \f$ as in \ref tensor2matrix .
  \param mat matrix M
  \param d1 dimension of space \f$ \mathbb{R}^n \f$
  \param d2 dimension of space \f$ \mathbb{R}^m \f$
  \param d3 dimension of space \f$ \mathbb{R}^p \f$
  \param d4 dimension of space \f$ \mathbb{R}^q \f$
  \return C
*/

//------------------------------------------------------------------------------------------------
template <class T> tensor4<T> matrix2tensor (const matrix<T>& mat, int d1, int d2, int d3, int d4)
//------------------------------------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());
  assert ( (d1*d2 == mat.Rows()) && (d3*d4 == mat.Columns()) );

  tensor4<T> tsr(d1,d2,d3,d4,false);

  for (int i=1,p=1; i<=tsr.dim1(); i++)
    for (int j=1; j<=tsr.dim2(); j++,p++)
      for (int k=1,q=1; k<=tsr.dim3(); k++)
	for (int l=1; l<=tsr.dim4(); l++)
	  tsr(i,j,k,l) = mat (p,q++);

  return tsr;
}


/*!
  \brief Converts a matrix into symmetric \f$ 4^\text{th} \f$ order tensor.

  Let \f$ M \f$ be a matrix in  \f$ \mathbb{R}^n \times \mathbb{R}^m \f$. We define \f$ C \f$ the \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^p \times \mathbb{R}^p \times \mathbb{R}^q \times \mathbb{R}^q \f$, with \f$ p = \frac{1}{2} (\sqrt{1+8n} - 1) \f$ and \f$ q = \frac{1}{2} (\sqrt{1+8m} - 1) \f$, such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$ and
  \f$ C_{ij} = \alpha_{ij} * M_{ij} \f$, where \n
  \f$ \alpha_{ij} = \left\{
      \begin{aligned}
      & 1 \text{ if } i=j \text{ and } k=l, \\
      & \frac{1}{2} \text{ if } i \neq j \text{ and } k \neq l, \\
      & \frac{1}{\sqrt{2}} \text{ elsewhere},
      \end{aligned}
  \right.
  \f$ \n
  and where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & \dots & n & n+1 & n+2 & \dots \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & \dots & nn & 12 & 13 & \dots \\
  \hline
  \end{array}
  \f$
  \param mat matrix M
  \return C
  \n\n
  \f$ \textbf{Example :} \text{ in dimension 3 - elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sqrt{2} \sigma_{12} \\ \sqrt{2} \sigma_{13} \\ \sqrt{2} \sigma_{23}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & \sqrt{2} C_{1112} & \sqrt{2} C_{1113} & \sqrt{2} C_{1123} \\
      C_{2211} & C_{2222} & C_{2233} & \sqrt{2} C_{2212} & \sqrt{2} C_{2213} & \sqrt{2} C_{2223} \\
      C_{3311} & C_{3322} & C_{3333} & \sqrt{2} C_{3312} & \sqrt{2} C_{3313} & \sqrt{2} C_{3323} \\
      \sqrt{2} C_{1211} & \sqrt{2} C_{1222} & \sqrt{2} C_{1233} & 2 C_{1212} & 2 C_{1213} & 2 C_{1223} \\
      \sqrt{2} C_{1311} & \sqrt{2} C_{1322} & \sqrt{2} C_{1333} & 2 C_{1312} & 2 C_{1313} & 2 C_{1323} \\
      \sqrt{2} C_{2311} & \sqrt{2} C_{2322} & \sqrt{2} C_{2333} & 2 C_{2312} & 2 C_{2313} & 2 C_{2323} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \sqrt{2} \EuScript{E}_{12} \\ \sqrt{2} \EuScript{E}_{13} \\ \sqrt{2} \EuScript{E}_{23}
      \end{pmatrix}
  \f$ \n
*/

//-------------------------------------------------------------------
template <class T> tensor4<T> matrix2symtensor (const matrix<T>& mat)
//-------------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());

  long double factor = .5 * sqrt( (long double) 2);
  int nb1 = .5 * (sqrt(1.+8*mat.Rows()) - 1);
  int nb2 = .5 * (sqrt(1.+8*mat.Columns()) - 1);
  tensor4<T> tsr(nb1,nb1,nb2,nb2,false);

  for (int i=1,p=nb1; i<=nb1; i++)
    for (int j=i,q=nb2; j<=nb1; j++,q=nb2)
      { if (i!=j) p++;
	for (int k=1; k<=nb2; k++)
	  for (int l=k; l<=nb2; l++)
	    if (i==j && k==l) tsr(i,i,k,k) = mat(i,k);
	    else if (i==j) tsr(i,i,k,l) = tsr(i,i,l,k) = factor * mat(i,++q);
	         else if (k==l) tsr(i,j,k,k) = tsr(j,i,k,k) = factor * mat(p,k);
		      else tsr(i,j,k,l) = tsr(j,i,k,l) = tsr(i,j,l,k) = tsr(j,i,l,k) = 0.5 * mat(p,++q);
      }

  return tsr;
}


/*!
  \brief Converts a \f$ 4^\text{th} \f$ order tensor into a matrix.

  Let \f$ C \f$ be a \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^n \times \mathbb{R}^m \times \mathbb{R}^p \times \mathbb{R}^q \f$. We define the matrix \f$ M \f$ in \f$ \mathbb{R}^{nm} \times \mathbb{R}^{pq} \f$ by
  \f$ M = 
      \begin{pmatrix}
      C_{1111} & C_{1112} & \dots  & C_{111p} & C_{1121} & \dots  & C_{11pq} \\
      C_{1211} & C_{1212} & \dots  & C_{121p} & C_{1221} & \dots  & C_{12pq} \\
      \vdots   & \vdots   & \ddots & \vdots   & \vdots   & \ddots & \vdots   \\
      C_{1m11} & C_{1m12} & \dots  & C_{1m1p} & C_{1m21} & \dots  & C_{1mpq} \\
      C_{2111} & C_{2112} & \dots  & C_{211p} & C_{2121} & \dots  & C_{21pq} \\
      \vdots   & \vdots   & \ddots & \vdots   & \vdots   & \ddots & \vdots   \\
      C_{nm11} & C_{nm12} & \dots  & C_{nm1p} & C_{nm21} & \dots  & C_{nmpq} \\
      \end {pmatrix}
  \f$
  \param tsr \f$ 4^\text{th} \f$ order tensor C
  \return M
*/

//----------------------------------------------------------------
template <class T> matrix<T> tensor2matrix (const tensor4<T>& tsr)
//----------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());

  matrix<T> mat(tsr.dim1()*tsr.dim2(),tsr.dim3()*tsr.dim4(),false);

  for (int i=1,p=1; i<=tsr.dim1(); i++)
    for (int j=1; j<=tsr.dim2(); j++,p++)
      for (int k=1,q=1; k<=tsr.dim3(); k++)
	for (int l=1; l<=tsr.dim4(); l++)
	  mat (p,q++) = tsr(i,j,k,l);

  return mat;
}


/*!
  \brief Converts a symmetric \f$ 2^\text{nd} \f$ order tensor into a vector.

  Let \f$ T \in \mathbb{R}^n \times \mathbb{R}^n \f$. 
  We define vector \f$ v \f$, with dimension \f$ \frac{1}{2} n (n+1) \f$, through \f$ T \f$ by \n
  \f$ T = \left( \begin{matrix}
      T_{11} = v_1 & T_{12} = \frac{1}{\sqrt{2}} v_{n+1} & T_{13} = \frac{1}{\sqrt{2}} v_{n+2} & \dots & T_{1n} = \frac{1}{\sqrt{2}} v_{2n-1} \\ \\
      T_{12} & T_{22} = v_2 & T_{23} = \frac{1}{\sqrt{2}} v_{2n} & \dots & T_{2n} = \frac{1}{\sqrt{2}} v_{3n-3} \\ \\
      T_{13} & T_{23} & T_{33} = v_3 & \dots & T_{3n} = \frac{1}{\sqrt{2}} v_{4n-6} \\ \\
      \vdots & \vdots & \vdots & \ddots & \vdots \\ \\
      T_{1n} & T_{2n} & T_{3n} & \dots & T_{nn} = v_n
      \end{matrix} \right) \f$.

  \param tsr symmetric \f$ 2^\text{nd} \f$ order tensor T
  \return vector \f$ v \f$
*/

//----------------------------------------------------------------------
template <class T> vector<T> symtensor2vector (const symtensor2<T>& tsr)
//----------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2());
  assert (tsr.dim1() == tsr.dim2());

  long double factor = sqrt( (long double) 2);
  int n = tsr.dim1();
  vector<T> v(.5*n*(n+1),false);

  for (int i=1,p=1; i<=n; i++)
    { v[i] = tsr(i,i);
      for (int j=i+1; j<=n; j++, p++)
	  v[n+p] = factor * tsr(i,j);
    }

  return v;
}


/*!
  \brief Converts a symmetric \f$ 4^\text{th} \f$ order tensor into a matrix.

  Let \f$ C \f$ be a \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^n \times \mathbb{R}^n \times \mathbb{R}^m \times \mathbb{R}^m \f$ such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$.
  We define the matrix \f$ M \in \mathbb{R}^{\frac{1}{2} n (n+1)} \times \mathbb{R}^{\frac{1}{2} m (m+1)} \f$ such as \n
  \f$ M_{ij} = \alpha_{ij} * C_{ij} \f$, where \n
  \f$ \alpha_{ij} = \left\{
      \begin{aligned}
      & 1 \text{ if } i=j \text{ and } k=l, \\
      & 2 \text{ if } i \neq j \text{ and } k \neq l, \\
      & \sqrt{2} \text{ elsewhere},
      \end{aligned}
  \right.
  \f$ \n
  and where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & \dots & n & n+1 & n+2 & \dots \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & \dots & nn & 12 & 13 & \dots \\
  \hline
  \end{array}
  \f$
  \param tsr \f$ 4^\text{th} \f$ order tensor C
  \return M
  \n\n
  \f$ \textbf{Example :} \text{ in dimension 3 - elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sqrt{2} \sigma_{12} \\ \sqrt{2} \sigma_{13} \\ \sqrt{2} \sigma_{23}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & \sqrt{2} C_{1112} & \sqrt{2} C_{1113} & \sqrt{2} C_{1123} \\
      C_{2211} & C_{2222} & C_{2233} & \sqrt{2} C_{2212} & \sqrt{2} C_{2213} & \sqrt{2} C_{2223} \\
      C_{3311} & C_{3322} & C_{3333} & \sqrt{2} C_{3312} & \sqrt{2} C_{3313} & \sqrt{2} C_{3323} \\
      \sqrt{2} C_{1211} & \sqrt{2} C_{1222} & \sqrt{2} C_{1233} & 2 C_{1212} & 2 C_{1213} & 2 C_{1223} \\
      \sqrt{2} C_{1311} & \sqrt{2} C_{1322} & \sqrt{2} C_{1333} & 2 C_{1312} & 2 C_{1313} & 2 C_{1323} \\
      \sqrt{2} C_{2311} & \sqrt{2} C_{2322} & \sqrt{2} C_{2333} & 2 C_{2312} & 2 C_{2313} & 2 C_{2323} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \sqrt{2} \EuScript{E}_{12} \\ \sqrt{2} \EuScript{E}_{13} \\ \sqrt{2} \EuScript{E}_{23}
      \end{pmatrix}
  \f$ \n
*/

//-------------------------------------------------------------------
template <class T> matrix<T> symtensor2matrix (const tensor4<T>& tsr)
//-------------------------------------------------------------------
{
  assert (tsr.dim1() && tsr.dim2() && tsr.dim3() && tsr.dim4());
  assert (tsr.dim1() == tsr.dim2());
  assert (tsr.dim3() == tsr.dim4());

  long double factor = sqrt( (long double) 2);
  int nb1 = tsr.dim1();
  int nb2 = tsr.dim3();
  matrix<T> mat(.5*nb1*(nb1+1),.5*nb2*(nb2+1),false);

  for (int i=1,p=nb1; i<=nb1; i++)
    for (int j=i,q=nb2; j<=nb1; j++,q=nb2)
      { if (i!=j) p++;
	for (int k=1; k<=nb2; k++)
	  for (int l=k; l<=nb2; l++)
	    { assert ( relative_error (tsr(i,j,k,l), tsr(i,j,l,k)) );
	      assert ( relative_error (tsr(i,j,k,l), tsr(j,i,k,l)) );
	      assert ( relative_error (tsr(i,j,k,l), tsr(j,i,l,k)) );
	      if (i==j && k==l) mat (i,k) = tsr(i,j,k,l);
	      else if (i==j) mat(i,++q) = factor * tsr(i,j,k,l);
	           else if (k==l) mat(p,k) = factor * tsr(i,j,k,l);
		        else mat(p,++q) = 2 * tsr(i,j,k,l);
	    }
      }

  return mat;
}


}


#endif
